<?php
/* Desenvolvido por Level Sistemas (http://www.levelsistemas.com.br)
** Contato: contato@levelsistemas.com.br */

defined('MOODLE_INTERNAL') || die();

function combinedview_supports($feature) {
    switch($feature) {
        case FEATURE_GROUPS:                  return true;
        case FEATURE_GROUPINGS:               return true;
        case FEATURE_MOD_INTRO:               return true;
        case FEATURE_COMPLETION_TRACKS_VIEWS: return true;
        case FEATURE_COMPLETION_HAS_RULES:    return true;
        case FEATURE_GRADE_HAS_GRADE:         return false;
        case FEATURE_GRADE_OUTCOMES:          return false;
        case FEATURE_BACKUP_MOODLE2:          return true;
        case FEATURE_SHOW_DESCRIPTION:        return true;
        case FEATURE_MOD_PURPOSE:             return MOD_PURPOSE_COMMUNICATION;

        default: return null;
    }
}

/**
 * This will create a new instance and return the id number
 * of the new instance.
 *
 * @global object
 * @param object $combinedview the object given by mod_combinedview_mod_form
 * @return int
 */
function combinedview_add_instance($combinedview) {
    global $DB;

    $combinedview->timemodified = time();
    $combinedview->id = '';

    $combinedviewid = $DB->insert_record("combinedview", $combinedview);

    $combinedview->id = $combinedviewid;

    if (isset($combinedview->selected_forms) && is_array($combinedview->selected_forms)) {
        $sortorder = 0;
        foreach ($combinedview->selected_forms as $formkey) {
            $parts = explode('_', $formkey, 2);
            if (count($parts) == 2) {
                $formtype = $parts[0];
                $formid = $parts[1];
                
                $formrecord = new stdClass();
                $formrecord->combinedviewid = $combinedviewid;
                $formrecord->formid = $formid;
                $formrecord->formtype = $formtype;
                $formrecord->sortorder = $sortorder++;
                $DB->insert_record("combinedview_forms", $formrecord);
            }
        }
    }

    return $combinedviewid;
}

function combinedview_update_instance($combinedview) {
    global $DB;

    $combinedview->timemodified = time();
    $combinedview->id = $combinedview->instance;

    $result = $DB->update_record("combinedview", $combinedview);

    if (isset($combinedview->selected_forms) && is_array($combinedview->selected_forms)) {
        $DB->delete_records("combinedview_forms", array("combinedviewid" => $combinedview->id));
        
        $sortorder = 0;
        foreach ($combinedview->selected_forms as $formkey) {
            $parts = explode('_', $formkey, 2);
            if (count($parts) == 2) {
                $formtype = $parts[0];
                $formid = $parts[1];
                
                $formrecord = new stdClass();
                $formrecord->combinedviewid = $combinedview->id;
                $formrecord->formid = $formid;
                $formrecord->formtype = $formtype;
                $formrecord->sortorder = $sortorder++;
                $DB->insert_record("combinedview_forms", $formrecord);
            }
        }
    }

    return $result;
}

function combinedview_delete_instance($id) {
    global $DB;

    if (!$combinedview = $DB->get_record("combinedview", array("id" => $id))) {
        return false;
    }

    $DB->delete_records("combinedview_forms", array("combinedviewid" => $id));

    $DB->delete_records("combinedview", array("id" => $combinedview->id));

    return true;
}

function combinedview_get_available_forms($courseid) {
    global $DB;

    $forms = array();

    $feedbackforms = $DB->get_records_sql(
        "SELECT f.id, f.name, 'feedback' as type, cm.id as cmid
         FROM {feedback} f
         JOIN {course_modules} cm ON cm.instance = f.id
         WHERE f.course = ? AND cm.module = (SELECT id FROM {modules} WHERE name = 'feedback')
         ORDER BY f.name",
        array($courseid)
    );

    foreach ($feedbackforms as $form) {
        $forms[$form->type . '_' . $form->id] = $form->name . ' (' . get_string('pluginname', 'mod_feedback') . ')';
    }

    $dataforms = $DB->get_records_sql(
        "SELECT d.id, d.name, 'data' as type, cm.id as cmid
         FROM {data} d
         JOIN {course_modules} cm ON cm.instance = d.id
         WHERE d.course = ? AND cm.module = (SELECT id FROM {modules} WHERE name = 'data')
         ORDER BY d.name",
        array($courseid)
    );

    foreach ($dataforms as $form) {
        $forms[$form->type . '_' . $form->id] = $form->name . ' (' . get_string('pluginname', 'mod_data') . ')';
    }

    if (get_config('mod_questionnaire', 'version')) {
        $questionnaireforms = $DB->get_records_sql(
            "SELECT q.id, q.name, 'questionnaire' as type, cm.id as cmid
             FROM {questionnaire} q
             JOIN {course_modules} cm ON cm.instance = q.id
             WHERE q.course = ? AND cm.module = (SELECT id FROM {modules} WHERE name = 'questionnaire')
             ORDER BY q.name",
            array($courseid)
        );

        foreach ($questionnaireforms as $form) {
            $forms[$form->type . '_' . $form->id] = $form->name . ' (' . get_string('pluginname', 'mod_questionnaire') . ')';
        }
    }

    $assignmentforms = $DB->get_records_sql(
        "SELECT a.id, a.name, 'assignment' as type, cm.id as cmid
         FROM {assign} a
         JOIN {course_modules} cm ON cm.instance = a.id
         WHERE a.course = ? AND cm.module = (SELECT id FROM {modules} WHERE name = 'assign')
         ORDER BY a.name",
        array($courseid)
    );

    foreach ($assignmentforms as $form) {
        $forms[$form->type . '_' . $form->id] = $form->name . ' (' . get_string('pluginname', 'mod_assign') . ')';
    }

    return $forms;
}

function combinedview_get_selected_forms($combinedviewid) {
    global $DB;

    // Validate input
    if (empty($combinedviewid) || !is_numeric($combinedviewid)) {
        return array();
    }

    return $DB->get_records_sql(
        "SELECT cf.*, cf.formtype as type, cf.formid
         FROM {combinedview_forms} cf
         WHERE cf.combinedviewid = ?
         ORDER BY cf.sortorder",
        array($combinedviewid)
    );
}

function combinedview_render_form($type, $formid, $courseid) {
    global $CFG, $DB, $OUTPUT;

    switch ($type) {
        case 'feedback':
            return combinedview_render_feedback_form($formid, $courseid);
        case 'data':
            return combinedview_render_data_form($formid, $courseid);
        case 'questionnaire':
            return combinedview_render_questionnaire_form($formid, $courseid);
        case 'assignment':
            return combinedview_render_assignment_form($formid, $courseid);
        default:
            return '';
    }
}

function combinedview_render_feedback_form($formid, $courseid) {
    global $CFG, $DB, $OUTPUT, $USER;

    require_once($CFG->dirroot . '/mod/feedback/lib.php');

    $feedback = $DB->get_record('feedback', array('id' => $formid));
    if (!$feedback) {
        return '';
    }

    $cm = get_coursemodule_from_instance('feedback', $formid);
    $context = context_module::instance($cm->id);

    // Check if user can complete this feedback
    if (!has_capability('mod/feedback:complete', $context)) {
        return '';
    }

    // Get feedback items
    $items = $DB->get_records('feedback_item', array('feedback' => $formid), 'position');
    
    if (empty($items)) {
        return '';
    }

    $output = '';
    $output .= html_writer::start_tag('div', array('class' => 'combinedview-form feedback-form'));
    $output .= html_writer::tag('h3', $feedback->name);
    
    // Create simple form fields for each item
    foreach ($items as $item) {
        $output .= html_writer::start_tag('div', array('class' => 'feedback-item'));
        
        // Use item name or label, fallback to item id
        $label = !empty($item->name) ? $item->name : 'Item ' . $item->id;
        $output .= html_writer::tag('label', $label);
        
        switch ($item->typ) {
            case 'textfield':
                $output .= html_writer::tag('input', '', array(
                    'type' => 'text',
                    'name' => 'feedback_' . $formid . '_' . $item->id,
                    'class' => 'form-control'
                ));
                break;
                
            case 'textarea':
                $output .= html_writer::tag('textarea', '', array(
                    'name' => 'feedback_' . $formid . '_' . $item->id,
                    'class' => 'form-control',
                    'rows' => 4
                ));
                break;
                
            case 'numeric':
                $output .= html_writer::tag('input', '', array(
                    'type' => 'number',
                    'name' => 'feedback_' . $formid . '_' . $item->id,
                    'class' => 'form-control'
                ));
                break;
                
            case 'multichoice':
                $options = combinedview_get_feedback_options($item);
                if ($options) {
                    $subtype = combinedview_get_feedback_subtype($item);
                    $inputtype = ($subtype == 'c') ? 'checkbox' : 'radio';
                    $inputname = 'feedback_' . $formid . '_' . $item->id;
                    if ($subtype == 'c') {
                        $inputname .= '[]';
                    }
                    
                    foreach ($options as $key => $option) {
                        $output .= html_writer::start_tag('div', array('class' => 'form-check'));
                        $output .= html_writer::tag('input', '', array(
                            'type' => $inputtype,
                            'name' => $inputname,
                            'value' => $key,
                            'class' => 'form-check-input'
                        ));
                        $output .= html_writer::tag('label', $option, array('class' => 'form-check-label'));
                        $output .= html_writer::end_tag('div');
                    }
                } else {
                    // Fallback to text input if no options found
                    $output .= html_writer::tag('input', '', array(
                        'type' => 'text',
                        'name' => 'feedback_' . $formid . '_' . $item->id,
                        'class' => 'form-control'
                    ));
                }
                break;
                
            case 'multichoicerated':
                $options = combinedview_get_feedback_options($item);
                if ($options) {
                    $subtype = combinedview_get_feedback_subtype($item);
                    $inputtype = ($subtype == 'c') ? 'checkbox' : 'radio';
                    $inputname = 'feedback_' . $formid . '_' . $item->id;
                    if ($subtype == 'c') {
                        $inputname .= '[]';
                    }
                    
                    foreach ($options as $key => $option) {
                        $output .= html_writer::start_tag('div', array('class' => 'form-check'));
                        $output .= html_writer::tag('input', '', array(
                            'type' => $inputtype,
                            'name' => $inputname,
                            'value' => $key,
                            'class' => 'form-check-input'
                        ));
                        $output .= html_writer::tag('label', $option, array('class' => 'form-check-label'));
                        $output .= html_writer::end_tag('div');
                    }
                } else {
                    // Fallback to text input if no options found
                    $output .= html_writer::tag('input', '', array(
                        'type' => 'text',
                        'name' => 'feedback_' . $formid . '_' . $item->id,
                        'class' => 'form-control'
                    ));
                }
                break;
                
            case 'info':
                // Info items are just display, no input needed
                $output .= html_writer::tag('div', format_text($item->presentation, FORMAT_HTML), array('class' => 'feedback-info'));
                break;
                
            case 'label':
                // Label items are just display, no input needed
                $output .= html_writer::tag('div', format_text($item->presentation, FORMAT_HTML), array('class' => 'feedback-label'));
                break;
                
            default:
                $output .= html_writer::tag('input', '', array(
                    'type' => 'text',
                    'name' => 'feedback_' . $formid . '_' . $item->id,
                    'class' => 'form-control'
                ));
        }
        
        $output .= html_writer::end_tag('div');
    }
    
    $output .= html_writer::end_tag('div');

    return $output;
}

/**
 * Get feedback options from item presentation
 */
function combinedview_get_feedback_options($item) {
    if (empty($item->presentation)) {
        return array();
    }
    
    $presentation = $item->presentation;
    
    // Check if it's a multichoice type
    if (strpos($presentation, '>>>>>') !== false) {
        $parts = explode('>>>>>', $presentation);
        if (count($parts) > 1) {
            $optionspart = $parts[1];
            // Remove horizontal adjustment if present
            if (strpos($optionspart, '<<<<<') !== false) {
                $optionspart = explode('<<<<<', $optionspart)[0];
            }
            $options = explode('|', $optionspart);
            $result = array();
            foreach ($options as $idx => $option) {
                $result[$idx + 1] = format_text(trim($option), FORMAT_HTML, array('noclean' => true, 'para' => false));
            }
            return $result;
        }
    }
    
    return array();
}

/**
 * Get feedback subtype from item presentation
 */
function combinedview_get_feedback_subtype($item) {
    if (empty($item->presentation)) {
        return 'r'; // Default to radio
    }
    
    $presentation = $item->presentation;
    
    if (strpos($presentation, '>>>>>') !== false) {
        $parts = explode('>>>>>', $presentation);
        return $parts[0];
    }
    
    return 'r'; // Default to radio
}

function combinedview_render_assignment_form($formid, $courseid) {
    global $CFG, $DB, $OUTPUT, $USER;

    require_once($CFG->dirroot . '/mod/assign/lib.php');

    $assignment = $DB->get_record('assign', array('id' => $formid));
    if (!$assignment) {
        return '';
    }

    $cm = get_coursemodule_from_instance('assign', $formid);
    $context = context_module::instance($cm->id);

    // Check if user can submit to this assignment
    if (!has_capability('mod/assign:submit', $context)) {
        return '';
    }

    // Check if assignment is open for submissions
    $now = time();
    if (!empty($assignment->allowsubmissionsfromdate) && $now < $assignment->allowsubmissionsfromdate) {
        return '';
    }
    if (!empty($assignment->duedate) && $now > $assignment->duedate + $assignment->cutoffdate) {
        return '';
    }

    // Create assignment instance
    require_once($CFG->dirroot . '/mod/assign/locallib.php');
    $assign = new assign($context, $cm, $assignment->course);

    $output = '';
    $output .= html_writer::start_tag('div', array('class' => 'combinedview-form assignment-form'));
    $output .= html_writer::tag('h3', $assignment->name);
    
    // Add assignment description
    if (!empty($assignment->intro)) {
        $output .= html_writer::start_tag('div', array('class' => 'assignment-intro'));
        $output .= format_module_intro('assign', $assignment, $cm->id);
        $output .= html_writer::end_tag('div');
    }

    // Add submission form fields
    $output .= html_writer::start_tag('div', array('class' => 'assignment-submission'));
    
    // Check submission plugins configuration
    $submissionplugins = $assign->get_submission_plugins();
    $onlinetext_enabled = false;
    $file_enabled = false;
    
    foreach ($submissionplugins as $plugin) {
        if ($plugin->get_type() === 'onlinetext' && $plugin->is_enabled()) {
            $onlinetext_enabled = true;
        }
        if ($plugin->get_type() === 'file' && $plugin->is_enabled()) {
            $file_enabled = true;
        }
    }
    
    // Text submission (if enabled)
    if ($onlinetext_enabled) {
        $output .= html_writer::start_tag('div', array('class' => 'assignment-field'));
        $output .= html_writer::tag('label', get_string('submission_text', 'combinedview'));
        $output .= html_writer::tag('textarea', '', array(
            'name' => 'assignment_' . $formid . '_onlinetext',
            'class' => 'form-control',
            'rows' => 6,
            'placeholder' => get_string('submission_text_placeholder', 'combinedview')
        ));
        $output .= html_writer::end_tag('div');
    }

    // File submission (if enabled)
    if ($file_enabled) {
        $output .= html_writer::start_tag('div', array('class' => 'assignment-field'));
        $output .= html_writer::tag('label', get_string('submission_files', 'combinedview'));
        $output .= html_writer::tag('input', '', array(
            'type' => 'file',
            'name' => 'assignment_' . $formid . '_files[]',
            'class' => 'form-control',
            'multiple' => 'multiple'
        ));
        $output .= html_writer::tag('small', get_string('submission_files_help', 'combinedview'), array('class' => 'form-text text-muted'));
        $output .= html_writer::end_tag('div');
    }

    // Comments
    $output .= html_writer::start_tag('div', array('class' => 'assignment-field'));
    $output .= html_writer::tag('label', get_string('submission_comments', 'combinedview'));
    $output .= html_writer::tag('textarea', '', array(
        'name' => 'assignment_' . $formid . '_comments',
        'class' => 'form-control',
        'rows' => 3,
        'placeholder' => get_string('submission_comments_placeholder', 'combinedview')
    ));
    $output .= html_writer::end_tag('div');

    $output .= html_writer::end_tag('div');
    $output .= html_writer::end_tag('div');

    return $output;
}

function combinedview_render_data_form($formid, $courseid) {
    global $CFG, $DB, $OUTPUT, $USER;

    require_once($CFG->dirroot . '/mod/data/lib.php');

    $data = $DB->get_record('data', array('id' => $formid));
    if (!$data) {
        return '';
    }

    $cm = get_coursemodule_from_instance('data', $formid);
    $context = context_module::instance($cm->id);

    // Check if user can add entries
    if (!has_capability('mod/data:writeentry', $context)) {
        return '';
    }

    // Get data fields
    $fields = $DB->get_records('data_fields', array('dataid' => $formid), 'sortorder');
    
    if (empty($fields)) {
        return '';
    }

    $output = '';
    $output .= html_writer::start_tag('div', array('class' => 'combinedview-form data-form'));
    $output .= html_writer::tag('h3', $data->name);
    
    // Create form fields
    foreach ($fields as $field) {
        $fieldobj = data_get_field($field, $data);
        if ($fieldobj) {
            $output .= html_writer::start_tag('div', array('class' => 'data-field'));
            $output .= html_writer::tag('label', $field->name);
            $output .= $fieldobj->display_add_field();
            $output .= html_writer::end_tag('div');
        }
    }
    
    $output .= html_writer::end_tag('div');

    return $output;
}

function combinedview_render_questionnaire_form($formid, $courseid) {
    global $CFG, $DB, $OUTPUT, $USER;

    // Check if questionnaire module is available
    if (!get_config('mod_questionnaire', 'version')) {
        return '';
    }

    require_once($CFG->dirroot . '/mod/questionnaire/lib.php');

    $questionnaire = $DB->get_record('questionnaire', array('id' => $formid));
    if (!$questionnaire) {
        return '';
    }

    $cm = get_coursemodule_from_instance('questionnaire', $formid);
    $context = context_module::instance($cm->id);

    // Check if user can respond
    if (!has_capability('mod/questionnaire:submit', $context)) {
        return '';
    }

    // Get questionnaire questions
    $questions = $DB->get_records('questionnaire_question', array('survey_id' => $questionnaire->sid), 'position');
    
    if (empty($questions)) {
        return '';
    }

    $output = '';
    $output .= html_writer::start_tag('div', array('class' => 'combinedview-form questionnaire-form'));
    $output .= html_writer::tag('h3', $questionnaire->name);
    
    // Create form fields for questions
    foreach ($questions as $question) {
        $output .= html_writer::start_tag('div', array('class' => 'questionnaire-question'));
        $output .= html_writer::tag('label', $question->content);
        // Add appropriate input field based on question type
        $output .= html_writer::tag('input', '', array('type' => 'text', 'name' => 'q_' . $question->id));
        $output .= html_writer::end_tag('div');
    }
    
    $output .= html_writer::end_tag('div');

    return $output;
}

function combinedview_process_submission($combinedviewid, $formdata) {
    global $DB, $USER;

    $combinedview = $DB->get_record('combinedview', array('id' => $combinedviewid));
    if (!$combinedview) {
        return false;
    }

    $selectedforms = combinedview_get_selected_forms($combinedviewid);
    $success = true;

    foreach ($selectedforms as $form) {
        $type = $form->type;
        $formid = $form->formid;

        switch ($type) {
            case 'feedback':
                $success &= combinedview_process_feedback_submission($formid, $formdata);
                break;
            case 'data':
                $success &= combinedview_process_data_submission($formid, $formdata);
                break;
            case 'questionnaire':
                $success &= combinedview_process_questionnaire_submission($formid, $formdata);
                break;
            case 'assignment':
                $success &= combinedview_process_assignment_submission($formid, $formdata);
                break;
        }
    }

    return $success;
}

function combinedview_process_feedback_submission($formid, $formdata) {
    global $CFG, $DB, $USER;

    require_once($CFG->dirroot . '/mod/feedback/lib.php');

    $feedback = $DB->get_record('feedback', array('id' => $formid));
    if (!$feedback) {
        return false;
    }

    $cm = get_coursemodule_from_instance('feedback', $formid);
    $context = context_module::instance($cm->id);

    // Check if user can complete this feedback
    if (!has_capability('mod/feedback:complete', $context)) {
        return false;
    }

    // Create feedback completion record
    $completed = new stdClass();
    $completed->feedback = $formid;
    $completed->userid = $USER->id;
    $completed->guestid = '';
    $completed->timemodified = time();
    $completed->anonymous_response = $feedback->anonymous;

    $completedid = $DB->insert_record('feedback_completed', $completed);

    // Process each feedback item
    $items = $DB->get_records('feedback_item', array('feedback' => $formid));
    
    foreach ($items as $item) {
        $fieldname = 'feedback_' . $formid . '_' . $item->id;
        
        // Skip info and label items as they don't have input values
        if (in_array($item->typ, array('info', 'label'))) {
            continue;
        }
        
        if (isset($formdata[$fieldname])) {
            $value = new stdClass();
            $value->completed = $completedid;
            $value->item = $item->id;
            
            // Handle different question types
            if (in_array($item->typ, array('multichoice', 'multichoicerated'))) {
                $subtype = combinedview_get_feedback_subtype($item);
                if ($subtype == 'c') {
                    // Checkbox - multiple values
                    if (is_array($formdata[$fieldname])) {
                        $value->value = implode('|', $formdata[$fieldname]);
                    } else {
                        $value->value = $formdata[$fieldname];
                    }
                } else {
                    // Radio or dropdown - single value
                    $value->value = $formdata[$fieldname];
                }
            } else {
                // Other types (textfield, textarea, numeric)
                $value->value = $formdata[$fieldname];
            }
            
            $DB->insert_record('feedback_value', $value);
        }
    }

    return true;
}

function combinedview_process_data_submission($formid, $formdata) {
    global $CFG, $DB, $USER;

    require_once($CFG->dirroot . '/mod/data/lib.php');

    $data = $DB->get_record('data', array('id' => $formid));
    if (!$data) {
        return false;
    }

    $cm = get_coursemodule_from_instance('data', $formid);
    $context = context_module::instance($cm->id);

    // Check if user can add entries
    if (!has_capability('mod/data:writeentry', $context)) {
        return false;
    }

    // Create new entry
    $entry = new stdClass();
    $entry->dataid = $formid;
    $entry->userid = $USER->id;
    $entry->groupid = 0;
    $entry->timecreated = time();
    $entry->timemodified = time();

    $entryid = $DB->insert_record('data_records', $entry);

    // Process field data
    $fields = $DB->get_records('data_fields', array('dataid' => $formid));
    
    foreach ($fields as $field) {
        $fieldobj = data_get_field($field, $data);
        if ($fieldobj) {
            $fieldobj->update_content($entryid, $formdata, $field);
        }
    }

    return true;
}

function combinedview_process_questionnaire_submission($formid, $formdata) {
    global $CFG, $DB, $USER;

    // Check if questionnaire module is available
    if (!get_config('mod_questionnaire', 'version')) {
        return false;
    }

    require_once($CFG->dirroot . '/mod/questionnaire/lib.php');

    $questionnaire = $DB->get_record('questionnaire', array('id' => $formid));
    if (!$questionnaire) {
        return false;
    }

    $cm = get_coursemodule_from_instance('questionnaire', $formid);
    $context = context_module::instance($cm->id);

    // Check if user can respond
    if (!has_capability('mod/questionnaire:submit', $context)) {
        return false;
    }

    // Create response
    $response = new stdClass();
    $response->survey_id = $questionnaire->sid;
    $response->userid = $USER->id;
    $response->submitted = time();

    $responseid = $DB->insert_record('questionnaire_response', $response);

    // Process question responses
    $questions = $DB->get_records('questionnaire_question', array('survey_id' => $questionnaire->sid));
    
    foreach ($questions as $question) {
        $questionkey = 'q_' . $question->id;
        if (isset($formdata[$questionkey])) {
            $answer = new stdClass();
            $answer->response_id = $responseid;
            $answer->question_id = $question->id;
            $answer->choice_id = 0; // For text questions
            $answer->response = $formdata[$questionkey];
            
            $DB->insert_record('questionnaire_response_text', $answer);
        }
    }

    return true;
}

function combinedview_process_assignment_submission($formid, $formdata) {
    global $CFG, $DB, $USER;

    require_once($CFG->dirroot . '/mod/assign/lib.php');
    require_once($CFG->dirroot . '/mod/assign/locallib.php');

    $assignment = $DB->get_record('assign', array('id' => $formid));
    if (!$assignment) {
        return false;
    }

    $cm = get_coursemodule_from_instance('assign', $formid);
    $context = context_module::instance($cm->id);

    // Check if user can submit to this assignment
    if (!has_capability('mod/assign:submit', $context)) {
        return false;
    }

    // Check if assignment is open for submissions
    $now = time();
    if (!empty($assignment->allowsubmissionsfromdate) && $now < $assignment->allowsubmissionsfromdate) {
        return false;
    }
    if (!empty($assignment->duedate) && $now > $assignment->duedate + $assignment->cutoffdate) {
        return false;
    }

    // Create assignment instance
    $assign = new assign($context, $cm, $assignment->course);

    // Get or create submission
    $submission = $assign->get_user_submission($USER->id, true);

    // Process text submission
    $textfieldname = 'assignment_' . $formid . '_onlinetext';
    if (isset($formdata[$textfieldname]) && !empty($formdata[$textfieldname])) {
        // Update text submission directly in database
        $onlinetext = $DB->get_record('assignsubmission_onlinetext', array('submission' => $submission->id));
        if ($onlinetext) {
            $onlinetext->onlinetext = $formdata[$textfieldname];
            $onlinetext->onlineformat = FORMAT_HTML;
            $DB->update_record('assignsubmission_onlinetext', $onlinetext);
        } else {
            $onlinetext = new stdClass();
            $onlinetext->submission = $submission->id;
            $onlinetext->assignment = $formid;
            $onlinetext->onlinetext = $formdata[$textfieldname];
            $onlinetext->onlineformat = FORMAT_HTML;
            $DB->insert_record('assignsubmission_onlinetext', $onlinetext);
        }
    }

    // Process comments
    $commentsfieldname = 'assignment_' . $formid . '_comments';
    if (isset($formdata[$commentsfieldname]) && !empty($formdata[$commentsfieldname])) {
        $submission->submissioncomment = $formdata[$commentsfieldname];
        $DB->update_record('assign_submission', $submission);
    }

    // Update submission status
    $submission->status = ASSIGN_SUBMISSION_STATUS_SUBMITTED;
    $submission->timemodified = time();
    $DB->update_record('assign_submission', $submission);

    // Process file uploads (if any)
    $filesfieldname = 'assignment_' . $formid . '_files';
    if (isset($_FILES[$filesfieldname]) && !empty($_FILES[$filesfieldname]['name'][0])) {
        // Handle file uploads
        $fs = get_file_storage();
        $component = 'assignsubmission_file';
        $filearea = 'submission_files';
        
        foreach ($_FILES[$filesfieldname]['name'] as $key => $filename) {
            if (!empty($filename)) {
                $fileinfo = array(
                    'contextid' => $context->id,
                    'component' => $component,
                    'filearea' => $filearea,
                    'itemid' => $submission->id,
                    'filepath' => '/',
                    'filename' => $filename
                );
                
                $fs->create_file_from_pathname($fileinfo, $_FILES[$filesfieldname]['tmp_name'][$key]);
            }
        }
    }

    return true;
}

function combinedview_get_view_actions() {
    return array('view', 'view all');
}

function combinedview_get_post_actions() {
    return array('submit');
}

function combinedview_reset_userdata($data) {
    return array();
}

function combinedview_get_coursemodule_info($coursemodule) {
    global $DB;

    $combinedview = $DB->get_record('combinedview', array('id' => $coursemodule->instance), 'id, name, intro, introformat');
    
    if ($combinedview) {
        $info = new cached_cm_info();
        $info->name = $combinedview->name;
        if ($coursemodule->showdescription) {
            $info->content = format_module_intro('combinedview', $combinedview, $coursemodule->id, false);
        }
        return $info;
    }
    
    return null;
} 