<?php

use local_avaliador\utils;

global $CFG;
require_once($CFG->libdir . '/formslib.php');

class user_link_form extends moodleform
{
    public function definition()
    {
        $mform = $this->_form;
        $data = $this->_customdata;

        $associated_users = $data['associated_users'] ?? [];
        $userid = $data['userid'] ?? 0;

        $showAvaliadores = true;
        if ($userid) {
            $this->add_hidden_user_field($mform, $userid);
        } else {
            $showAvaliadores = $this->add_user_selection_field($mform);
        }

        if ($showAvaliadores) {
            $this->add_associated_users_field($mform, $associated_users);
            $this->add_action_buttons();
        } else {
            $mform->addElement('cancel');
        }
    }

    /**
     * Adiciona o campo de usuário principal como campo oculto e congelado.
     *
     * @param MoodleQuickForm $mform
     * @param int $userid
     */
    private function add_hidden_user_field(MoodleQuickForm $mform, int $userid): void
    {
        $mform->addElement('hidden', 'userid', $userid);
        $mform->setType('userid', PARAM_INT);

        $userName = $this->get_user_name($userid);
        $mform->addElement('static', 'userid_display', get_string('usuarioprincipal', 'local_avaliador'), $userName);
    }

    /**
     * Adiciona o campo de seleção de usuário principal, baseado no papel configurado.
     *
     * @param MoodleQuickForm $mform
     * @return bool
     */
    private function add_user_selection_field(MoodleQuickForm $mform): bool
    {
        $role_default = get_config('local_avaliador', 'role_default');
        $users = $this->get_users_by_role($role_default);

        if (empty($users)) {
            $mform->addElement('static', 'userid', get_string('selectuser', 'local_avaliador'), '<i style="color: red">Nenhum usuário disponível para o papel selecionado.</i>');
            return false;
        }

        $mform->addElement('select', 'userid', get_string('selectuser', 'local_avaliador'), $users);
        $mform->setType('userid', PARAM_INT);
        $mform->addRule('userid', get_string('required'), 'required', null, 'client');
        return true;
    }

    /**
     * Adiciona o campo para selecionar usuários associados (avaliadores).
     *
     * @param MoodleQuickForm $mform
     * @param array $associated_users
     */
    private function add_associated_users_field(MoodleQuickForm $mform, array $associated_users): void
    {
        global $PAGE, $OUTPUT;

        $all_potential_users = $this->get_all_potential_avaliadores();

        $selected_ids = array_flip($associated_users);
        $all_users_data = is_array($all_potential_users) ? $all_potential_users : [];

        $options_selected = [];
        $options_available = [];
        foreach ($all_users_data as $id => $name) {
            if (isset($selected_ids[$id])) {
                $options_selected[$id] = $name;
            } else {
                $options_available[$id] = $name;
            }
        }

        $select_attrs = ['multiple' => 'multiple', 'size' => 15];

        $el_available = $mform->createElement(
            'select',
            'avaliadores_available',
            get_string('available', 'core'),
            $options_available,
            $select_attrs + ['id' => 'id_avaliadores_available']
        );

        $el_selected = $mform->createElement(
            'select',
            'avaliadorid',
            get_string('selected', 'core'),
            $options_selected,
            $select_attrs + ['id' => 'id_avaliadores_selected']
        );
        $el_selected->setMultiple(true); // <-- ESSENCIAL
        $mform->setType('avaliadorid', PARAM_INT);

        $mform->addElement('hidden', 'avaliadorid_send', '');
        $mform->setType('avaliadorid_send', PARAM_RAW);

        $btn_right_text = '&gt;';
        $btn_all_right_text = '&gt;&gt;';
        $btn_left_text = '&lt;';
        $btn_all_left_text = '&lt;&lt;';

        $button_html = '
        <div class="dual-select-buttons d-flex flex-column justify-content-center mx-2">
            <button type="button" id="btn_move_right" class="btn btn-secondary mb-1" title="' . get_string('add') . '">' . $btn_right_text . '</button>
            <button type="button" id="btn_move_all_right" class="btn btn-secondary mb-1" title="' . get_string('addall') . '">' . $btn_all_right_text . '</button>
            <button type="button" id="btn_move_left" class="btn btn-secondary mb-1" title="' . get_string('remove') . '">' . $btn_left_text . '</button>
            <button type="button" id="btn_move_all_left" class="btn btn-secondary" title="' . get_string('removeall') . '">' . $btn_all_left_text . '</button>
        </div>';
        $el_buttons = $mform->createElement('html', $button_html);

        $group_elements = [$el_available, $el_buttons, $el_selected];
        $mform->addGroup($group_elements, 'avaliadorid_group', get_string('selectlinkedusers', 'local_avaliador'), '', false);

        $mform->addElement('html', '<style>
             #fitem_id_avaliadorid_group > .felement { display: flex; align-items: center; }
             #fitem_id_avaliadorid_group select { min-width: 200px; }
         </style>');

        $formid = $this->_form->getAttribute('id');

        $js = "
        require(['core/event'], function(event) {

            function setupManualDualSelect(formId, availableSelId, selectedSelId) {
                const form = document.getElementById(formId);
                const availableList = document.getElementById(availableSelId);
                const selectedList = document.getElementById(selectedSelId);
                const btnRight = document.getElementById('btn_move_right');
                const btnAllRight = document.getElementById('btn_move_all_right');
                const btnLeft = document.getElementById('btn_move_left');
                const btnAllLeft = document.getElementById('btn_move_all_left');

                if (!form || !availableList || !selectedList || !btnRight || !btnAllRight || !btnLeft || !btnAllLeft) {
                    console.warn('Elementos do Dual Select Manual não encontrados para o formulário:', formId);
                    return;
                }

                const moveOptions = (source, dest, all = false) => {
                    const options = source.options;
                    for (let i = options.length - 1; i >= 0; i--) {
                        if (all || options[i].selected) {
                            options[i].selected = false;
                            dest.appendChild(options[i]);
                        }
                    }
                };

                btnRight.addEventListener('click', () => moveOptions(availableList, selectedList));
                btnAllRight.addEventListener('click', () => moveOptions(availableList, selectedList, true));
                btnLeft.addEventListener('click', () => moveOptions(selectedList, availableList));
                btnAllLeft.addEventListener('click', () => moveOptions(selectedList, availableList, true));

                form.addEventListener('submit', function() {
                    const selectedOptions = selectedList.options;
                    for (let i = 0; i < selectedOptions.length; i++) {
                        selectedOptions[i].selected = true;
                    }
                        const avaliadoridSend = form.querySelector('[name=\"avaliadorid_send\"]');
                        if (avaliadoridSend) {
                            avaliadoridSend.value = Array.from(selectedOptions)
                                .map(option => option.value)
                                .join('|');
                        }
                });
            }

            setupManualDualSelect('{$formid}', 'id_avaliadores_available', 'id_avaliadores_selected');

        });
        ";

        $PAGE->requires->js_init_code($js);
    }

    /**
     * Obtém o nome completo de um usuário pelo ID.
     *
     * @param int $userid
     * @return string
     */
    private function get_user_name(int $userid): string
    {
        global $DB;
        $user = $DB->get_record('user', ['id' => $userid], 'id, firstname, lastname, deleted');

        if ($user && !$user->deleted) {
            return "{$user->firstname} {$user->lastname}";
        }
        return get_string('usernotfound');
    }

    /**
     * Obtém todos os usuários do sistema, exceto os já vinculados.
     *
     * @return array
     */
    private function get_all_potential_avaliadores(): array
    {
        global $DB;
        $sql = "SELECT u.id, CONCAT(u.firstname, ' ', u.lastname) AS fullname
                  FROM {user} u
                 WHERE u.deleted = 0 AND u.suspended = 0 AND u.confirmed = 1
              ORDER BY u.firstname, u.lastname";

        $users = $DB->get_records_sql_menu($sql);
        return $users ?: [];
    }

    /**
     * Obtém usuários baseados em um papel específico, excluindo os já vinculados.
     *
     * @param int $roleId
     * @return array
     */
    private function get_users_by_role($roleId)
    {
        global $DB;

        $sql = "SELECT u.id, CONCAT(u.firstname, ' ', u.lastname) AS name
                  FROM {user} u
                  JOIN {role_assignments} ra ON ra.userid = u.id
                  LEFT JOIN {local_avaliador_users_links} l ON l.userid = u.id
                 WHERE l.userid IS NULL
              ORDER BY u.firstname";

        return $DB->get_records_sql_menu($sql);
    }
}
